<?php
/**
 *
 * Let's Review Frontend Class
 *
 * @since      1.0.0
 *
 * @package    Let's Review
 * @subpackage lets-review/frontend
 */

class Lets_Review_Frontend {

	/**
	 * Var for Let's Review slug.
	 *
	 * @since    1.0.0
	 */
	private $slug;

	/**
	 * Var for Let's Review version.
	 *
	 * @since    1.0.0
	 */
	private $version;

	/**
	 * Var for Let's Review URL.
	 *
	 * @since    1.0.0
	 */
	private $url;

	/**
	 * Frontend Constructor
	 *
	 * @since 1.0.0
	 *
	*/
	public function __construct( $slug = '', $version = '', $url = '' ) {
		$this->slug    = $slug;
		$this->version = $version;
		$this->url     = $url;
	}

	public function lets_review_enqueue_assets() {
		wp_enqueue_style( 'lets-review-api', $this->url . 'assets/css/style-api.min.css', array(), $this->version, 'all' );
		if ( get_option( 'lets_review_gen_fa', 1 ) == 1 && apply_filters( 'lets_review_fa_off', true ) == true ) {
			wp_enqueue_style( 'font-awesome', $this->url . 'assets/fonts/fontawesome/css/fontawesome.min.css', array(), $this->version, 'all' );
			wp_enqueue_style( 'font-awesome-regular', $this->url . 'assets/fonts/fontawesome/css/regular.min.css', array(), $this->version, 'all' );
			wp_enqueue_style( 'font-awesome-solid', $this->url . 'assets/fonts/fontawesome/css/solid.min.css', array(), $this->version, 'all' );
		}
		$loader = self::asset_optimizer();
		if ( empty( $loader ) ) {
			return;
		}
		$this->lets_review_enqueue_scripts_frontend();
		$this->lets_review_enqueue_styles_frontend();
	}

	/**
	 * Load Let's Review scripts frontend
	 *
	 * @since    1.0.0
	 */
	public function lets_review_enqueue_scripts_frontend() {
		if ( is_front_page() ) {
			$zeen = class_exists( 'ZeenBlocks' );
		}
		$dep = array( 'jquery', 'js-cookie' );
		wp_enqueue_script( 'js-cookie', $this->url . 'assets/js/js.cookie.min.js', array(), '2.2.1', true );
		if ( get_option( 'lets_review_gen_lb', 1 ) == 1 && is_singular() && empty( $zeen ) ) {
			wp_dequeue_script( 'photoswipe' );
			wp_dequeue_script( 'photoswipe-ui' );
			wp_deregister_script( 'photoswipe' );
			wp_deregister_script( 'photoswipe-ui' );
			wp_dequeue_style( 'photoswipe' );
			wp_deregister_style( 'photoswipe' );
			wp_dequeue_style( 'photoswipe-default-skin' );
			wp_deregister_style( 'photoswipe-default-skin' );
			wp_enqueue_style( 'photoswipe', $this->url . 'assets/css/photoswipe.min.css', array(), '4.1.3' );
			wp_enqueue_script( 'photoswipe-ui', $this->url . 'assets/js/photoswipe-ui-default.min.js', array(), '4.1.3', true );
			$dep[] = 'photoswipe-ui';
			wp_enqueue_script( 'photoswipe', $this->url . 'assets/js/photoswipe.min.js', array(), '4.1.3', true );
			$dep[] = 'photoswipe';
		}
		$with_user_reviews = '';
		if ( is_singular() ) {
			global $post;
			if ( ( get_option( 'lets_review_gen_user_reviews' ) == 1 || 10 == get_post_meta( $post->ID, '_lets_review_type', true ) ) && get_post_meta( $post->ID, '_lets_review_onoff', true ) == 1 ) {
				$with_user_reviews = true;
				wp_enqueue_script( 'jquery-ui-slider' );
				wp_enqueue_script( 'jquery-ui-touch-punch', $this->url . 'assets/js/jquery.ui.touch-punch.min.js', array( 'jquery-ui-slider' ), '0.2.3', true );
			}
		}

		$js_ext = get_option( 'lets_review_gen_js_min', 1 ) != 1 ? '' : '.min';
		wp_enqueue_script( 'lets-review-functions', esc_url( $this->url . 'assets/js/functions' . $js_ext . '.js' ), $dep, $this->version, true );
		wp_localize_script(
			'lets-review-functions',
			'lrJS',
			array(
				'root'                 => esc_url_raw( rest_url() ) . 'codetipi-lets-review/v3/',
				'nonce'                => wp_create_nonce( 'wp_rest' ),
				'i18n'                 => $this->lets_review_i18n(),
				'letsReviewAUrl'       => admin_url( 'admin-ajax.php' ),
				'letsReviewNonce'      => wp_create_nonce( 'lets_review_nonce' ),
				'letsReviewUReviews'   => $with_user_reviews,
				'intersectionObserver' => $this->url . 'assets/js/intersection-observer.min.js',
			)
		);

	}
	function lets_review_i18n() {
		$output                 = array();
		$output['alreadyVoted'] = esc_html__( 'Change Your Rating', 'lets-review' );
		return $output;
	}
	/**
	 * Let's Review Fonts
	 *
	 * @since    1.0.0
	*/
	public function lets_review_enqueue_styles_frontend() {

		$ssl          = is_ssl() ? 'https' : 'http';
		$headings     = get_option( 'lets_review_gen_type_headings' );
		$font_heading = ( empty( $headings ) || $headings == 'none' ) ? '' : $headings;
		$para         = get_option( 'lets_review_gen_type_p' );
		$font_p       = ( empty( $para ) || $para == 'none' ) ? '' : $para;
		$ext          = get_option( 'lets_review_gen_type_ext' );
		$ext          = empty( $ext ) ? '' : $ext;

		$font_heading_url = ! empty( $font_heading ) ? str_replace( ' ', '+', $font_heading ) . ':400,700' : '';
		$p_url            = ( ! empty( $font_p ) && $font_p != $font_heading ) ? '|' . str_replace( ' ', '+', $font_p ) . ':400,700,400italic' : '';

		$ext_url = '';
		if ( ! empty( $ext ) ) {
			$ext_url .= '&subset=';
			foreach ( $ext as $key => $value ) {
				$ext_url .= $value . ',';
			}
			$ext_url = rtrim( $ext_url, ',' );
		}

		if ( ! empty( $p_url ) || ! empty( $font_heading_url ) ) {
			wp_enqueue_style( 'lets-review-fonts', $ssl . '://fonts.googleapis.com/css?family=' . $font_heading_url . $p_url . $ext_url );
		}

		wp_enqueue_style( 'lets-review', $this->url . 'assets/css/style.min.css', array(), $this->version, 'all' );
		wp_add_inline_style( 'lets-review', $this->lets_review_extra_style() );
	}

	/**
	 * Let's Review Custom CSS
	 *
	 * @since    1.0.0
	 */
	public function lets_review_extra_style() {
		$headings     = get_option( 'lets_review_gen_type_headings' );
		$font_heading = ( empty( $headings ) || 'Inherit from Theme' == $headings ) ? '' : get_option( 'lets_review_gen_type_headings' );
		$para         = get_option( 'lets_review_gen_type_p' );
		$font_p       = ( empty( $para ) || get_option( 'lets_review_gen_type_p' ) == 'Inherit from Theme' ) ? '' : get_option( 'lets_review_gen_type_p' );

		$output = '';
		if ( ! empty( $font_heading ) && 'none' != $font_heading ) {
			$output .= '.cb-lr-font-heading, .lr-font-h { font-family: ' . esc_attr( $font_heading ) . '; }';
		}

		if ( ! empty( $font_p ) && 'none' != $font_p ) {
			$output .= '.cb-lr-font-p, .lr-font-p, .lets-review-block__wrap { font-family: ' . esc_attr( $font_p ) . '; }';
		}
		return $output;
	}

	/**
	 * Let's Review User Rating
	 *
	 * @since    1.0.0
	 */
	public function lets_review_user_rating( $pid = '', $args = '' ) {
		global $post;

		if ( empty( $args['sc'] ) ) {
			$pid        = empty( $pid ) ? $post->ID : $pid;
			$uid        = '';
			$design     = get_post_meta( $pid, '_lets_review_design', true );
			$design     = ! empty( $design ) ? $design : 1;
			$format     = get_post_meta( $pid, '_lets_review_format', true );
			$type       = get_post_meta( $pid, '_lets_review_type', true );
			$custom_img = get_post_meta( $post->ID, '_lets_review_custom_image', true );
			$icon       = get_post_meta( $pid, '_lets_review_custom_icon', true );

		} else {
			$pid        = $args['postid'];
			$uid        = $args['id'];
			$design     = $args['design'];
			$format     = $args['format'];
			$custom_img = $args['custom_image'];
			$icon       = html_entity_decode( $args['icon_code'] );
			$type       = $args['type'];
		}

		$vote_count  = get_post_meta( $pid, '_lets_review_user_rating_vote_count' . $uid, true ) ? get_post_meta( $pid, '_lets_review_user_rating_vote_count' . $uid, true ) : 0;
		$user_rating = get_post_meta( $pid, '_lets_review_user_rating' . $uid, true );
		$design_ani  = get_post_meta( $pid, '_lets_review_design_ani', true );
		$design_ani  = ! empty( $design_ani ) ? $design_ani : 1;
		$color       = $this->lets_review_color( $pid );

		if ( empty( $user_rating ) ) {
			$user_rating = 0;
			$score       = 0;
			$width       = 0;
		}

		$best_rating = 5;
		$classes     = '';
		$bg_style    = '';

		if ( ! empty( $args ) ) {
			$design  = $design < 3 ? $design : 1;
			$classes = ' lets-review-uid-rating cb-skin-base cb-no-fi lets-review-ani-' . $design_ani . ' cb-design-' . $design . ' cb-score-type-' . $format;
			if ( $format > 2 ) {
				$classes .= ' cb-out-of-5';
				$bg_style = 'style="background-color: ' . $color . ';"';
			}
		}
		$width = $user_rating;
		switch ( $format ) {
			case '1':
				$score       = $user_rating;
				$best_rating = 100;
				$icon_output = '<span class="cb-overlay score-overlay-wrap"><span style="width:' . (int) ( $width ) . '%; background-color: ' . $color . ';"></span></span>';
				break;
			case '2':
				$score       = $user_rating / 10;
				$best_rating = 10;
				$icon_output = '<span class="cb-overlay score-overlay-wrap"><span style="width:' . (int) ( $width ) . '%; background-color: ' . $color . ';" class="score-overlay"></span></span>';
				break;
			case '3':
				$score       = $user_rating / 20;
				$width_rev   = 100 - $width;
				$icon_output = '<span class="cb-overlay score-overlay-wrap" style="color: ' . $color . ';">' . str_repeat( '<i class="' . apply_filters( 'lets_review_stars', 'fas fa-star' ) . '"></i>', 5 ) . '<span style="width:' . (int) ( $width_rev ) . '%;" class="score-overlay"></span></span>';
				break;
			case '4':
				$score       = $user_rating / 20;
				$width_rev   = 100 - $width;
				$icon_output = '<span class="cb-overlay score-overlay-wrap" style="color: ' . $color . ';">' . str_repeat( $icon, 5 ) . '<span style="width:' . (int) ( $width_rev ) . '%;" class="score-overlay"></span></span>';
				break;

			case '5':
				$uimage      = $custom_img ? wp_get_attachment_image( $custom_img, 'thumbnail' ) : '';
				$score       = $user_rating / 20;
				$width_rev   = 100 - $width;
				$icon_output = '<span class="cb-overlay score-overlay-wrap">' . str_repeat( $uimage, 5 ) . '<span style="width:' . (int) ( $width_rev ) . '%;" class="score-overlay"></span></span>';
				break;
			default:
				$score       = $user_rating / 20;
				$width_rev   = 100 - $width;
				$icon_output = '<span class="cb-overlay score-overlay-wrap" style="color: ' . $color . ';">' . str_repeat( '<i class="' . apply_filters( 'lets_review_stars', 'fas fa-star' ) . '"></i>', 5 ) . '<span style="width:' . (int) ( $width_rev ) . '%;" class="score-overlay"></span></span>';
				break;
		}
		$reader_text       = get_option( 'lets_review_gd_reader_text' );
		$title             = empty( $reader_text ) ? esc_html__( 'Reader Rating', 'lets-review' ) : $reader_text;
		$user_rating_title = esc_html__( 'Leave Rating', 'lets-review' );
		$vote_txt          = sprintf( _n( 'Vote', 'Votes', $vote_count, 'lets-review' ), $vote_count );
		$cb_tip_title      = 'data-title="' . esc_html__( 'You have already rated', 'lets-review' ) . '"';

		if ( empty( $uid ) ) {
			$voted = isset( $_COOKIE['lets_review_user_rating'] ) && in_array( $pid, json_decode( $_COOKIE['lets_review_user_rating'] ) ) ? ' cb-rated lr-tip lr-tip-move' : '';
		} else {
			$voted = isset( $_COOKIE['lets_review_user_rating'] ) && in_array( $uid, json_decode( $_COOKIE['lets_review_user_rating'] ) ) ? ' cb-rated lr-tip lr-tip-move' : '';
		}

		if ( get_option( 'lets_review_gen_rating_permission' ) == 2 && ! is_user_logged_in() ) {
			$voted        = ' cb-rated lr-tip lr-tip-move';
			$cb_tip_title = 'data-title="' . esc_html__( 'You need to be logged in to vote', 'lets-review' ) . '"';
		}

		$output = '<div id="cb-user-rating-' . (int) $pid . '" class="cb-user-rating-block lr-cf' . $voted . $classes . '" ' . $cb_tip_title . '><div class="cb-bar"><span class="cb-criteria" data-cb-txt="' . esc_attr( $user_rating_title ) . '">' . $title . '<span class="cb-votes-count"><span>' . $vote_count . '</span> ' . $vote_txt . '</span></span><span class="cb-criteria-score" data-cb-score-cache="' . floatval( $score ) . '" data-cb-width-cache="' . floatval( $width ) . '">' . $score . '</span>' . $icon_output . '</div></div>';

		return $output;
	}

	/**
	 * Let's Review User Rating
	 *
	 * @since    1.0.0
	 */
	public function lets_review_a_cb() {

		if ( ! wp_verify_nonce( $_POST['letsReviewNonce'], 'lets_review_nonce' ) ) {
			die();
		}

		$score  = floatval( $_POST['letsReviewScore'] );
		$pid    = floatval( $_POST['letsReviewPostID'] );
		$uid    = empty( $_POST['letsReviewUID'] ) ? '' : floatval( $_POST['letsReviewUID'] );
		$format = empty( $_POST['letsReviewFormat'] ) ? get_post_meta( $pid, '_lets_review_format', true ) : (int) ( $_POST['letsReviewFormat'] );

		$votes_cache = get_post_meta( $pid, '_lets_review_user_rating_vote_count' . $uid, true );
		$votes_cache = empty( $votes_cache ) ? 0 : (int) ( $votes_cache );
		$score_cache = (int) ( get_post_meta( $pid, '_lets_review_user_rating' . $uid, true ) );
		$votes       = $votes_cache + 1;

		if ( $votes_cache == 1 ) {
			$score = (int) ( $score_cache + $score ) / 2;
		} elseif ( $votes_cache > 1 ) {
			$score_cache_total = $score_cache * $votes_cache;
			$score             = (int) ( ( $score_cache_total + $score ) / $votes );
		}

		$cook   = isset( $_COOKIE['lets_review_user_rating'] ) ? json_decode( $_COOKIE['lets_review_user_rating'], true ) : array();
		$cook[] = empty( $uid ) ? $pid : $uid;

		update_post_meta( $pid, '_lets_review_user_rating' . $uid, (int) ( $score ) );
		update_post_meta( $pid, '_lets_review_user_rating_vote_count' . $uid, (int) ( $votes ) );

		switch ( $format ) {
			case 1:
				$score_output = round( $score );
				break;
			case 2:
				$score_output = $score / 10;
				break;

			default:
				$score_output = ( number_format( ( $score / 20 ), 1 ) + 0 );
				$score        = 100 - $score;
				break;
		}

		$vote_count = get_post_meta( $pid, '_lets_review_user_rating_vote_count' . $uid, true ) ? get_post_meta( $pid, '_lets_review_user_rating_vote_count' . $uid, true ) : '0';
		$vote_txt   = sprintf( _n( 'Vote', 'Votes', $vote_count, 'lets-review' ), $vote_count );

		die( wp_json_encode( array( $score, $score_output, $vote_count . ' ' . $vote_txt, $cook ) ) );
	}

	/**
	 * Let's Review Gallery
	 *
	 * @since    1.0.0
	 */
	public function lets_review_media_gallery( $pid = '', $args = '' ) {

		global $post;

		if ( empty( $pid ) ) {
			$pid = $post->ID;
		}
		if ( empty( $args['sc'] ) ) {
			$imgs  = get_post_meta( $pid, '_lets_review_gallery_imgs', true );
			$title = get_post_meta( $pid, '_lets_review_gallery_title', true );
		} else {
			if ( ! empty( $args['gallery'] ) ) {
				$initial_img = explode( ',', $args['gallery'] );
				$imgs        = array();
				foreach ( $initial_img as $key ) {
					 $imgs[] = array( 'attachment-id' => $key );
				}
			}
			$title = empty( $args['gallery-title'] ) ? '' : $args['gallery-title'];
		}
		if ( ! empty( $imgs ) ) {
			$output = '';

			if ( ! empty( $title ) ) {
				$output = '<div class="cb-gallery-title cb-title cb-block-header cb-lr-font-heading cb-font-header">' . $title . '</diV>';
			}
			$output  .= '<div id="cb-review-gallery-wrap" class="cb-review-gallery lr-cf">';
			$lb_class = get_option( 'lets_review_gen_lb', 1 ) == 0 ? get_option( 'lets_review_gen_lb_custom_class' ) : '';
			foreach ( $imgs as $key ) {
				if ( isset( $key['attachment-id'] ) ) {
					$attachment = wp_get_attachment_image_src( $key['attachment-id'], 'full' );
					$img_title  = get_the_title( $key['attachment-id'] ) ? ' title="' . get_the_title( $key['attachment-id'] ) . '"' : '';
					$output    .= '<a href="' . esc_url( $attachment[0] ) . '" class="lets-review-lightbox ' . esc_attr( $lb_class ) . ' cb-lr-img cb-img-ani-1" rel="gal-' . $pid . '"' . $img_title . '>' . wp_get_attachment_image( $key['attachment-id'], 'thumbnail' ) . '</a>';
				}
			}
			$output .= '</div>';

			return $output;
		}
	}

	/**
	 * Let's Review Featured Image
	 *
	 * @since    1.0.0
	 */
	public function lets_review_media_fi( $pid = '', $design = '', $args = '' ) {
		global $post;

		if ( empty( $pid ) ) {
			$pid = $post->ID;
		}
		$source = get_post_meta( $pid, '_lets_review_media_use_fi', true );
		$fi     = 1 == $source ? get_post_thumbnail_id( $pid ) : get_post_meta( $pid, '_lets_review_fi', true );

		if ( empty( $args['sc'] ) ) {
			$design = empty( $design ) ? get_post_meta( $pid, '_lets_review_design', true ) : $design;
		} else {
			$design = $args['design'];
			$fi     = empty( $args['main_image'] ) ? $fi : $args['main_image'];
		}

		if ( ! empty( $fi ) ) {
			if ( 3 == $design || 4 == $design ) {
					return '<div class="cb-review-fi lets-review-full-bg" style=" background-image: url( ' . esc_url( wp_get_attachment_url( $fi ) ) . ' ) ; "></div>';
			} else {
				 $lb_class  = get_option( 'lets_review_gen_lb', 1 ) == 0 ? get_option( 'lets_review_gen_lb_custom_class' ) : '';
				$attachment = wp_get_attachment_image_src( $fi, 'full' );
				$img_title  = get_the_title( $fi ) ? ' title="' . get_the_title( $fi ) . '"' : '';
				return '<div class="cb-review-fi"><a href="' . esc_url( $attachment[0] ) . '" class="lets-review-lightbox ' . esc_attr( $lb_class ) . ' cb-lr-img cb-img-ani-1" rel="gal-fi-' . $pid . '"' . $img_title . '>' . wp_get_attachment_image( $fi, 'thumbnail' ) . '</a></div>';
			}
		}
	}

	/**
	 * Let's Review Color
	 *
	 * @since    1.0.0
	 */
	public function lets_review_color( $pid = '', $args = '' ) {
		$color = empty( $args['accent'] ) ? get_post_meta( $pid, '_lets_review_color', true ) : $args['accent'];
		if ( empty( $color ) ) {
			$color = get_option( 'lets_review_gd_color', '#f8d92f' );
		}
		return $color;
	}

	/**
	 * Let's Review Criteria
	 *
	 * @since    1.0.0
	 */
	public function lets_review_criterias( $pid = '', $args = '' ) {
		global $post;

		if ( empty( $pid ) ) {
			$pid = $post->ID;
		}

		if ( empty( $args['sc'] ) ) {
			$options     = get_post_meta( $pid, '_lets_review_criterias', true );
			$format      = get_post_meta( $pid, '_lets_review_format', true );
			$design_ani  = get_post_meta( $pid, '_lets_review_design_ani', true );
			$custom_icon = get_post_meta( $pid, '_lets_review_custom_icon', true );
		} else {

			$format     = $args['format'];
			$design_ani = $args['animation'];

			if ( empty( $args['user_review_crits'] ) ) {

				if ( empty( $args['criterias'] ) ) {
					return;
				}

				$crit_array = explode( ',', $args['criterias'] );
				$options    = array();
				$i          = 1;
				foreach ( $crit_array as $key ) {
					if ( $i % 2 == 0 ) {
						switch ( $format ) {
							case '2':
								$key = $key / 10;
								break;
							case '1':
								break;
							default:
								$key = ( number_format( ( $key / 20 ), 1 ) + 0 );
								break;
						}
						$options[ $i - 1 ]['score'] = $key;
					} else {
						$options[ $i ] = array( 'title' => $key );
					}
					$i++;
				}
			} else {
				$options = $args['user_review_crits'];
			}
			$custom_icon = empty( $args['icon_code'] ) ? get_post_meta( $pid, '_lets_review_custom_icon', true ) : html_entity_decode( $args['icon_code'] );

		}

		$color = $this->lets_review_color( $pid, $args );

		if ( empty( $design_ani ) ) {
			$design_ani = 1;
		}

		$max = 1 == $format ? 100 : 5;
		$max = 2 == $format ? 10 : 5;
		if ( ! empty( $options ) ) {

			$output = '<div class="cb-criteria-block lets-review-unseen lr-cf">';

			foreach ( $options as $key ) {

				if ( empty( $key['score'] ) ) {
					$key['score'] = '0';
				}
				if ( is_array( $key['score'] ) ) {
					$key['score'] = $key['score'][0]; }
				if ( is_array( $key['title'] ) ) {
					$key['title'] = $key['title'][0]; }
				if ( $key['score'] > $max ) {
					$key['score'] = $max;
				}
				if ( $key['score'] < 0 ) {
					$key['score'] = 0;
				}
				$output .= '<div class="cb-bar"><span class="cb-criteria">' . $key['title'] . '</span>';

				switch ( $format ) {
					case '1':
						$width_score = $key['score'];
						$output     .= '<span class="cb-criteria-score">' . (float) $key['score'] . '</span>';
						$output     .= '<span class="cb-overlay score-overlay-wrap"><span style="width:' . $width_score . '%; background-color: ' . $color . ';" class="score-overlay"></span></span></div>';
						break;
					case '2':
						$width_score = $key['score'] * 10;
						$output     .= '<span class="cb-criteria-score">' . (float) $key['score'] . '</span>';
						$output     .= '<span class="cb-overlay score-overlay-wrap"><span style="width:' . $width_score . '%; background-color: ' . $color . ';" class="score-overlay"></span></span></div>';
						break;
					case '3':
						$output .= '<span class="cb-overlay score-overlay-wrap" style="color: ' . $color . ';">' . str_repeat( '<i class="' . apply_filters( 'lets_review_stars', 'fas fa-star' ) . '"></i>', 5 ) . '<span style="width:' . (int) ( 100 - ( $key['score'] * 20 ) ) . '%;" class="score-overlay"></span></span></div>';
						break;
					case '4':
						$output .= '<span class="cb-overlay score-overlay-wrap" style="color: ' . $color . ';">' . str_repeat( $custom_icon, 5 ) . '<span style="width:' . (int) ( 100 - ( $key['score'] * 20 ) ) . '%;" class="score-overlay"></span></span></div>';
						break;
					case '5':
						$custom_img = get_post_meta( $post->ID, '_lets_review_custom_image', true );
						$icon       = ! empty( $custom_img ) ? wp_get_attachment_image( $custom_img, 'thumbnail' ) : '';
						$output    .= '<span class="cb-overlay score-overlay-wrap">' . str_repeat( $icon, 5 ) . '<span style="width:' . (int) ( 100 - ( $key['score'] * 20 ) ) . '%;" class="score-overlay"></span></span></div>';
						break;

					default:
						$output .= '<span class="cb-overlay score-overlay-wrap" style="color: ' . $color . ';">' . str_repeat( '<i class="' . apply_filters( 'lets_review_stars', 'fas fa-star' ) . '"></i>', 5 ) . '<span style="width:' . (int) ( 100 - ( $key['score'] * 20 ) ) . '%;" class="score-overlay"></span></span></div>';
						break;
				}
			}

			$output .= '</div>';
			return $output;
		}

	}

	/**
	 * Let's Review Pros
	 *
	 * @since    1.0.0
	 */
	public function lets_review_pros( $pid = '', $args = '' ) {
		global $post;

		if ( empty( $pid ) ) {
			$pid = $post->ID;
		}

		if ( empty( $args['sc'] ) ) {
			$pros       = get_post_meta( $pid, '_lets_review_pros', true );
			$pros_title = get_post_meta( $pid, '_lets_review_pros_title', true );
		} else {
			if ( ! empty( $args['pros'] ) ) {
				$initial_pros = explode( apply_filters( 'lets_review_backend_delimiter', ',' ), $args['pros'] );
				$pros         = array();
				foreach ( $initial_pros as $key ) {
					 $pros[] = array( 'positive' => $key );
				}
			}
			$pros_title = empty( $args['pros_title'] ) ? '' : $args['pros_title'];
		}

		if ( ! empty( $pros ) ) {
			$output = '<div class="cb-pros">';
			if ( ! empty( $pros_title ) ) {
				$output .= '<div class="cb-pros-title cb-lr-font-heading cb-font-header cb-block-header lr-cf">' . $pros_title . '</div>';
			}

			foreach ( $pros as $key ) {
				if ( ! empty( $key['positive'] ) ) {
					$output .= '<div class="cb-pros-cons cb-pro">' . $key['positive'] . '</div>';
				}
			}

			$output .= '</div>';

			return $output;
		}

	}

	/**
	 * Let's Review Cons
	 *
	 * @since    1.0.0
	 */
	public function lets_review_cons( $pid = '', $args = '' ) {
		global $post;

		if ( empty( $pid ) ) {
			$pid = $post->ID;
		}

		if ( empty( $args['sc'] ) ) {
			$cons       = get_post_meta( $pid, '_lets_review_cons', true );
			$cons_title = get_post_meta( $pid, '_lets_review_cons_title', true );
		} else {
			if ( ! empty( $args['cons'] ) ) {
				$initial_cons = explode( apply_filters( 'lets_review_backend_delimiter', ',' ), $args['cons'] );
				$cons         = array();
				foreach ( $initial_cons as $key ) {
					$cons[] = array( 'negative' => $key );
				}
			}
			$cons_title = empty( $args['cons_title'] ) ? '' : $args['cons_title'];
		}

		if ( ! empty( $cons ) ) {
			$output = '<div class="cb-cons">';

			if ( ! empty( $cons_title ) ) {
				$output .= '<div class="cb-cons-title cb-lr-font-heading cb-font-header cb-block-header lr-cf">' . $cons_title . '</div>';
			}

			foreach ( $cons as $key ) {

				if ( ! empty( $key['negative'] ) ) {
					$output .= '<div class="cb-pros-cons cb-con">' . $key['negative'] . '</div>';
				}
			}
			$output .= '</div>';

			return $output;
		}

	}

	/**
	 * Let's Review Final Score
	 *
	 * @since    1.0.0
	 */
	public function lets_review_final_score( $pid = '', $score_el = '', $design = '', $args = '', $data = '' ) {
		global $post;

		if ( empty( $pid ) ) {
			$pid = $post->ID;
		}

		$output    = '';
		$core_type = 1;

		if ( empty( $args['sc'] ) ) {
			$f_score     = get_post_meta( $pid, '_lets_review_final_score', true );
			$type        = get_post_meta( $pid, '_lets_review_type', true );
			$score_sub   = get_post_meta( $pid, '_lets_review_subtitle', true );
			$format      = get_post_meta( $pid, '_lets_review_format', true );
			$design      = empty( $design ) ? get_post_meta( $pid, '_lets_review_design', true ) : $design;
			$design_ani  = get_post_meta( $pid, '_lets_review_design_ani', true ) ? get_post_meta( $pid, '_lets_review_design_ani', true ) : 1;
			$uid         = '';
			$custom_img  = get_post_meta( $post->ID, '_lets_review_custom_image', true );
			$custom_icon = get_post_meta( $pid, '_lets_review_custom_icon', true );
		} else {
			$type      = empty( $args['type'] ) ? 2 : $args['type'];
			$score_sub = empty( $args['score_subtitle'] ) ? '' : $args['score_subtitle'];
			$uid       = empty( $args['id'] ) ? '' : $args['id'];
			$f_score   = $args['final_score'];
			$core_type = 2;
			$format    = $args['format'];
			switch ( $format ) {
				case '2':
					$f_score = $f_score / 10;
					break;
				case '1':
					break;

				default:
					$f_score = ( number_format( ( $f_score / 20 ), 1 ) + 0 );
					break;
			}

			$design      = $args['design'];
			$design_ani  = $args['animation'];
			$custom_img  = empty( $args['custom_image'] ) ? '' : $args['custom_image'];
			$custom_icon = empty( $args['icon_code'] ) ? '' : html_entity_decode( $args['icon_code'] );

		}

		$color       = $this->lets_review_color( $pid, $args );
		$class       = '';
		$color_cache = '';
		$best_rating = 5;

		if ( ! empty( $score_el ) ) {
			$color_cache = 'background: ' . $color . ';';
			$color       = '';
		}

		if ( 3 == $type ) {
			$f_score = get_post_meta( $pid, '_lets_review_user_rating' . $uid, true ) ? get_post_meta( $pid, '_lets_review_user_rating' . $uid, true ) : '-';
		}
		$f_score = (float) $f_score + 0;
		if ( ! empty( $data ) ) {
			return array(
				'score'  => $f_score,
				'format' => $format,
			);
		}
		$f_score_cache = $f_score;

		if ( empty( $f_score ) && 0 != $f_score ) {
			return;
		}

		if ( '-' === $f_score ) {
			$class = ' cb-score--';
		} else {

			switch ( $format ) {
				case 1:
					$best_rating = 100;
					break;
				case 2:
					$f_score     = 3 == $type ? ( number_format( $f_score, 1 ) / 10 ) + 0 : ( number_format( $f_score, 1 ) + 0 );
					$best_rating = 10;
					break;
				case 3:
					$width          = 3 == $type ? 100 - $f_score : 100 - ( $f_score * 20 );
					$f_score_5_w    = 3 == $type ? number_format( ( $f_score / 20 ), 1 ) + 0 : $f_score;
					$f_score_number = get_option( 'lets_review_stars_f_score_w_number', 1 ) == 1 ? '<div class="lets-review-5-w-number">' . $f_score_5_w . '</div>' : '';
					$f_score        = $f_score_number . '<div class="cb-bar"><span class="cb-overlay score-overlay-wrap" style="color: ' . $color . ';">' . str_repeat( '<i class="' . apply_filters( 'lets_review_stars', 'fas fa-star' ) . '"></i>', 5 ) . '<span style="width:' . (int) ( $width ) . '%; ' . $color_cache . '" class="score-overlay"></span></span></div>';
					break;
				case 4:
					$width          = 3 == $type ? 100 - $f_score : 100 - ( $f_score * 20 );
					$f_score_5_w    = 3 == $type ? number_format( ( $f_score / 20 ), 1 ) + 0 : $f_score;
					$f_score_number = get_option( 'lets_review_stars_f_score_w_number', 1 ) == 1 ? '<div class="lets-review-5-w-number">' . $f_score_5_w . '</div>' : '';
					$f_score        = $f_score_number . '<div class="cb-bar"><span class="cb-overlay score-overlay-wrap" style="color: ' . $color . ';">' . str_repeat( $custom_icon, 5 ) . '<span style="width:' . (int) ( $width ) . '%; ' . $color_cache . '" class="score-overlay"></span></span></div>';
					break;
				case 5:
					$uimage         = ! empty( $custom_img ) ? wp_get_attachment_image( $custom_img, 'thumbnail' ) : '';
					$f_score_5_w    = 3 == $type ? number_format( ( $f_score / 20 ), 1 ) + 0 : $f_score;
					$f_score_number = get_option( 'lets_review_stars_f_score_w_number', 1 ) == 1 ? '<div class="lets-review-5-w-number">' . $f_score_5_w . '</div>' : '';
					$width          = 3 == $type ? 100 - $f_score : 100 - ( $f_score * 20 );
					$f_score        = $f_score_number . '<div class="cb-bar"><span class="cb-overlay score-overlay-wrap" style="color: ' . $color . ';">' . str_repeat( $uimage, 5 ) . '<span style="width:' . (int) ( $width ) . '%; ' . $color_cache . '" class="score-overlay"></span></span></div>';
					break;
			}
		}

		if ( ! empty( $score_el ) ) {
			$out_of_5 = $format > 2 ? ' cb-out-of-5 lets-review-ext-5' : '';
			$output   = '<span class="' . esc_attr( $out_of_5 ) . '">' . $f_score . '</span>';
		} else {
			if ( 3 == $design ) {
				$output  = '<div class="cb-score-box cb-lr-font-heading cb-font-header" style="border-color: ' . esc_attr( $color ) . '; "><div class="cb-score-titles-wrap">';
				$output .= '<div class="cb-final-score lr-cf">' . ( $f_score ) . '</div>';
			} elseif ( 4 == $design ) {
				$output  = '<div class="cb-score-box cb-lr-font-heading cb-font-header" style=" background: ' . esc_attr( $color ) . '; "><div class="cb-score-titles-wrap">';
				$output .= '<div class="cb-final-score lr-cf">' . ( $f_score ) . '</div>';
			} else {
				$output  = '<div class="cb-score-box cb-lr-font-heading cb-font-header' . esc_attr( $class ) . '" style="background-color: ' . esc_attr( $color ) . '; "><div class="cb-score-titles-wrap">';
				$output .= '<div class="cb-final-score lr-cf">' . ( $f_score ) . '</div>';
			}

			if ( ! empty( $score_sub ) ) {
				$output .= '<div class="cb-score-subtitle">' . $score_sub . '</div>';
			}

			$output .= '</div></div>';
		}
		return $output;
	}

	/**
	 * Let's Review Top Review Box
	 *
	 * @since    1.0.0
	 */
	public function lets_review_top_small_box( $pid = '' ) {
		global $post;

		if ( empty( $pid ) ) {
			$pid = $post->ID;
		}

		$out_of_5 = '';

		$format      = get_post_meta( $pid, '_lets_review_format', true );
		$design      = get_post_meta( $pid, '_lets_review_design', true ) ? get_post_meta( $pid, '_lets_review_design', true ) : 1;
		$design_ani  = get_post_meta( $pid, '_lets_review_design_ani', true ) ? get_post_meta( $pid, '_lets_review_design_ani', true ) : 1;
		$design_skin = get_post_meta( $pid, '_lets_review_design_skin', true ) ? get_post_meta( $pid, '_lets_review_design_skin', true ) : 1;
		$affiliates  = $this->lets_review_affiliates( $pid );
		$media_fi    = $this->lets_review_media_fi( $pid );

		if ( $format > 2 ) {
			$out_of_5 = ' cb-out-of-5';
		}
		if ( $media_fi == '' ) {
			$out_of_5 .= ' cb-no-fi';
		}
		if ( 5 == $design ) {
			$design      = $design . ' cb-design-2';
			$design_skin = 1;
		}

		$output = '<div id="cb-review-box-small-' . (int) $pid . '" class="lets-review-box-wrap cb-skin-base cb-lr-font-p cb-design-' . $design . ' cb-skin-version-' . $design_skin . $out_of_5 . ' cb-score-type-' . $format . ' lets-review-ani-' . $design_ani . ' lr-cf cb-review-box-top-fs">';
		if ( 3 == $design || 4 == $design ) {
			$output .= $media_fi;
		}
		$output .= '<div class="cb-review-block cb-review-pad lr-cf">' . $this->lets_review_final_score( $pid ) . '</div>';
		$output .= $affiliates ? '<div class="cb-review-block cb-review-pad cb-aff-block lr-cf">' . $affiliates . '</div>' : '';
		$output .= '</div>';
		return $output;
	}

	/**
	 * Let's Review Affiliate
	 *
	 * @since    1.0.0
	 */
	public function lets_review_affiliates( $pid = '', $args = '' ) {
		global $post;

		if ( empty( $pid ) ) {
			$pid = $post->ID;
		}

		if ( empty( $args['sc'] ) ) {
			$options = get_post_meta( $pid, '_lets_review_aff_buttons', true );
			$title   = get_post_meta( $pid, '_lets_review_aff_title', true );
		} else {
			if ( ! empty( $args['affiliate'] ) ) {
				$aff_array = explode( apply_filters( 'lets_review_backend_delimiter', ',' ), $args['affiliate'] );
				$options   = array();
				$i         = 1;
				foreach ( $aff_array as $key ) {
					if ( $i % 2 == 0 ) {
						$options[ $i - 1 ]['url'] = $key;
					} else {
						$options[ $i ] = array( 'title' => $key );
					}
					$i++;
				}
			}
			$title = empty( $args['affiliate_title'] ) ? '' : $args['affiliate_title'];
		}

		$color = $this->lets_review_color( $pid, $args );

		if ( ! empty( $options ) ) {
			$output = '';
			if ( ! empty( $title ) ) {
				$output = '<div class="cb-aff-title cb-block-header">' . $title . '</diV>';
			}
			$output       .= '<div class="cb-aff-buttons">';
			$shopping_cart = get_option( 'lets_review_gen_type_shopping_onoff' ) == 1 ? '<div class="custom-shopping-icon cb-button-ani-1">' . get_option( 'lets_review_gen_type_shopping_icon' ) . '</div>' : '<i class="' . esc_attr( apply_filters( 'lets_review_aff_icon_1_class', 'fas fa-shopping-cart' ) ) . ' cb-button-ani-1"></i>';
			foreach ( $options as $key ) {
				$output .= '<div class="cb-aff-button"><a href="' . esc_attr( do_shortcode( $key['url'] ) ) . '" rel="noreferrer noopener nofollow" target="_blank">' . $key['title'] . '<span class="cb-icon-wrap" style="background-color: ' . esc_attr( $color ) . ';">' . $shopping_cart . '<i class="' . esc_attr( apply_filters( 'lets_review_aff_icon_2_class', 'fa fa-long-arrow-right' ) ) . ' cb-button-ani-1-hid cb-button-ani-1"></i></span></a></div>';
			}
			$output .= '</div>';
			return $output;
		}
	}

	/**
	 * Let's Review Title
	 *
	 * @since    1.0.0
	 */
	public function lets_review_title( $pid = '', $countdown = '', $args = '' ) {
		global $post;

		if ( empty( $pid ) ) {
			$pid = $post->ID;
		}

		if ( empty( $args['sc'] ) ) {
			$title = get_post_meta( $pid, '_lets_review_title_use_post', true ) == 1 ? get_the_title( $pid ) : get_post_meta( $pid, '_lets_review_title', true );
		} else {
			$title = empty( $args['title'] ) ? '' : $args['title'];
		}

		if ( ! empty( $title ) ) {

			$output  = '<div class="cb-review-title cb-lr-font-heading cb-font-header">';
			$output .= empty( $countdown ) ? '' : '<div class="cb-countdown">' . $countdown . '</div>';
			$output .= '<a href="' . esc_url( get_the_permalink( $pid ) ) . '">' . $title . '</a></div>';
			return $output;

		}
	}

	/**
	 * Let's Review Conclusion
	 *
	 * @since    1.0.0
	 */
	public function lets_review_conclusion( $pid = '', $conclusion_show_title = true, $conclusion_show_read_more = '', $args = '' ) {
		global $post;

		if ( empty( $pid ) ) {
			$pid = $post->ID;
		}

		$output = '';
		if ( empty( $args['sc'] ) ) {
			$conclusion_title = get_post_meta( $pid, '_lets_review_conclusion_title', true );
			$conclusion       = get_post_meta( $pid, '_lets_review_conclusion', true );
		} else {
			$conclusion_title = empty( $args['conclusion_title'] ) ? '' : $args['conclusion_title'];
			$conclusion       = empty( $args['conclusion'] ) ? '' : $args['conclusion'];
		}

		$type = get_post_meta( $pid, '_lets_review_type', true );
		if ( ! empty( $conclusion_title ) && ! empty( $conclusion_show_title ) ) {
			$output .= '<div class="cb-review-conclusion-title cb-block-header lr-cf">' . $conclusion_title . '</div>';
		}

		if ( ! empty( $conclusion ) ) {

			$output .= '<div class="cb-review-conclusion lr-cf">';
			$output .= $conclusion;
			if ( empty( $conclusion_show_title ) ) {
				$output .= '<a href="' . get_permalink( $pid ) . '" class="cb-full-review">' . esc_html__( 'Read full review', 'lets-review' ) . ' <i class="fa fa-long-arrow-right"></i></a>';
			}
			$output .= '</div>';
		}
		return $output;
	}

	/**
	 * Let's Review Review Box
	 *
	 * @since    1.0.0
	 */
	public function lets_review_review_box( $args = '' ) {
		if ( empty( $args['pid'] ) ) {
			global $post;
			$pid = $post->ID;
		} else {
			$pid = $args['pid'];
		}
		$class = 'cb-skin-base lets-review-box-wrap cb-lr-font-p lr-cf';
		if ( empty( $args['sc'] ) ) {
			$design      = get_post_meta( $pid, '_lets_review_design', true ) ? get_post_meta( $pid, '_lets_review_design', true ) : 1;
			$format      = get_post_meta( $pid, '_lets_review_format', true );
			$design_ani  = get_post_meta( $pid, '_lets_review_design_ani', true ) ? get_post_meta( $pid, '_lets_review_design_ani', true ) : 1;
			$design_skin = get_post_meta( $pid, '_lets_review_design_skin', true ) ? get_post_meta( $pid, '_lets_review_design_skin', true ) : 1;
			$location    = get_post_meta( $pid, '_lets_review_location', true );
			$type        = get_post_meta( $pid, '_lets_review_type', true );
		} else {
			$design               = $args['design'];
			$format               = $args['format'];
			$design_skin          = $args['skin'];
			$design_ani           = $args['animation'];
			$args['custom_image'] = empty( $args['custom_image'] ) ? '' : $args['custom_image'];
			$args['icon_code']    = empty( $args['icon_code'] ) ? '' : html_entity_decode( $args['icon_code'] );
			$location             = 1;
			$type                 = 2;
		}

		$title       = $this->lets_review_title( $pid, '', $args );
		$media_fi    = $this->lets_review_media_fi( $pid, '', $args );
		$conclusion  = $this->lets_review_conclusion( $pid, true, '', $args );
		$affiliates  = $this->lets_review_affiliates( $pid, $args );
		$color       = $this->lets_review_color( $pid, $args );
		$cons        = $this->lets_review_cons( $pid, $args );
		$pros        = $this->lets_review_pros( $pid, $args );
		$f_score     = $this->lets_review_final_score( $pid, '', '', $args );
		$gallery     = $this->lets_review_media_gallery( $pid, $args );
		$out_of_5    = '';
		$bg_style    = '';
		$author_id   = get_post_field( 'post_author', $pid );
		$design_skin = 4 == $design || 5 == $design ? 1 : $design_skin;
		if ( $format > 2 ) {
			$out_of_5 = ' cb-out-of-5';
			$bg_style = 'style="background-color: ' . $color . ';"';
		}
		$media_exists = empty( $media_fi ) ? ' cb-no-fi' : '';
		$design       = 5 == $design ? $design . ' cb-design-2' : $design;
		$user_rating  = 2 == $type ? '' : $this->lets_review_user_rating( $pid );
		$criterias    = 3 == $type ? '' : $criterias = $this->lets_review_criterias( $pid, $args );

		switch ( $design ) {
			case 2:
				$output  = $media_fi || $title ? '<div class="cb-review-block cb-review-pad cb-round-fi cb-title-block lr-cf" ' . $bg_style . '>' . $media_fi . $title . '</div>' : '';
				$output .= $criterias ? '<div class="cb-review-block cb-criteria-block-wrap lr-cf">' . $criterias . '</div>' : '';
				$output .= $user_rating ? '<div class="cb-review-block">' . $user_rating . '</div>' : '';
				$output .= $pros || $cons ? '<div class="cb-review-block cb-review-pad cb-pros-cons-block lr-cf">' . $pros . $cons . '</div>' : '';
				$output .= $conclusion || $f_score ? '<div class="cb-review-block cb-review-pad cb-conclusion-block lr-cf">' . $f_score . $conclusion . '</div>' : '';
				$output .= $gallery ? '<div class="cb-review-block cb-review-in-title cb-review-pad cb-gallery-block cb-gallery-block-center lr-cf">' . $gallery . '</div>' : '';
				$output .= $affiliates ? '<div class="cb-review-block cb-review-pad cb-aff-block lr-cf">' . $affiliates . '</div>' : '';
				break;
			case 3:
				$output  = $title || $pros || $cons || $media_fi || $f_score || $conclusion ? '<div class="cb-review-block cb-review-pad cb-summary-block lr-cf">' . $title : '';
				$output .= $media_fi;
				$output .= $conclusion ? '<div class="cb-conclusion-block lr-cf">' . $conclusion . '</div>' : '';
				$output .= $pros || $cons ? '<div class="cb-pros-cons-block lr-cf">' . $pros . $cons . '</div>' : '';
				$output .= $f_score ? '<div class="cb-fi-block lr-cf">' . $f_score . '</div>' : '';
				$output .= $title || $pros || $cons || $media_fi || $f_score || $conclusion ? '</div>' : '';
				$output .= $gallery ? '<div class="cb-review-block cb-review-in-title cb-gallery-block cb-gallery-block-center cb-review-pad-no-top lr-cf">' . $gallery . '</div>' : '';
				$output .= $affiliates ? '<div class="cb-review-block cb-review-pad-no-top cb-aff-block lr-cf">' . $affiliates . '</div>' : '';
				break;
			case 4:
				$output  = $title || $pros || $cons || $media_fi || $f_score || $conclusion ? '<div class="cb-review-block cb-summary-block lr-cf">' : '';
				$output .= $f_score ? '<div class="cb-fi-block lr-cf">' . $f_score . '</div>' : '';
				$output .= $media_fi;
				$output .= $title;
				$output .= $gallery ? '<div class="cb-review-in-title cb-gallery-block lr-cf">' . $gallery . '</div>' : '';
				$output .= $conclusion ? '<div class="cb-conclusion-block lr-cf">' . $conclusion . '</div>' : '';
				$output .= $pros || $cons ? '<div class="cb-pros-cons-block lr-cf">' . $pros . $cons . '</div>' : '';
				$output .= $title || $pros || $cons || $media_fi || $f_score || $conclusion ? '</div>' : '';
				$output .= $affiliates ? '<div class="cb-aff-block lr-cf">' . $affiliates . '</div>' : '';
				break;
			case 5:
				$output  = $pros || $cons ? '<div class="cb-review-block cb-review-pad cb-pros-cons-block lr-cf">' . $pros . $cons . '</div>' : '';
				$output .= $f_score ? '<div class="cb-fi-block lr-cf">' . $f_score . '</div>' : '';
				$output .= $affiliates ? '<div class="cb-review-block cb-aff-block lr-cf">' . $affiliates . '</div>' : '';
				break;
			default:
				$output  = $media_fi || $title ? '<div class="cb-review-block cb-review-pad cb-round-fi cb-summary-block lr-cf">' . $media_fi . $title . '</div>' : '';
				$output .= $conclusion || $gallery ? '<div class="cb-review-block cb-review-pad cb-conclusion-block lr-cf">' : '';
				$output .= $conclusion;
				$output .= $gallery ? '<div class="cb-review-in-title cb-gallery-block lr-cf">' . $gallery . '</div>' : '';
				$output .= $conclusion || $gallery ? '</div>' : '';
				$output .= $criterias || $user_rating ? '<div class="cb-review-block cb-criteria-block lets-review-unseen cb-review-pad lr-cf">' . $criterias : '';
				$output .= $user_rating;
				$output .= $criterias || $user_rating ? '</div>' : '';
				$output .= $pros || $cons || $f_score ? '<div class="cb-review-block cb-review-pad cb-score-pros lr-cf">' : '';
				$output .= $pros || $cons ? '<div class="cb-pros-cons-block lr-cf">' . $pros . $cons . '</div>' : '';
				$output .= $f_score;
				$output .= $pros || $cons || $f_score ? '</div>' : '';
				$output .= $affiliates ? '<div class="cb-review-block cb-review-pad cb-aff-block lr-cf">' . $affiliates . '</div>' : '';
				break;
		}
		$class .= ' cb-design-' . $design;
		$class .= ' cb-skin-version-' . $design_skin;
		$class .= $out_of_5;
		$class .= $media_exists;
		$class .= ' cb-score-type-' . $format;
		$class .= ' lets-review-ani-' . $design_ani;
		$class .= ' cb-location-' . $location;
		return '<div id="lets-review-box-' . (int) $pid . '" class="' . esc_attr( $class ) . '" data-pid="' . (int) $pid . '" data-format="' . (int) $format . '" data-type="' . (int) $type . '">' . apply_filters( 'lets_review_box_start', '' ) . $output . apply_filters( 'lets_review_box_end', '' ) . '</div>';
	}
	/**
	 * Let's Review User Rating Box
	 *
	 * @since    2.0.0
	 */
	public function lets_review_user_rating_box( $args = '' ) {
		if ( empty( $args ) ) {
			return;
		}
		$pid       = $args['postid'];
		$uid       = $args['id'];
		$score_box = ! empty( $args['score_box'] ) && $args['score_box'] == 'on' ? 1 : '';
		$title     = empty( $args['title'] ) ? '' : '<div class="cb-review-title cb-lr-font-heading cb-font-header">' . $args['title'] . '</div>';
		$content   = empty( $args['content_title'] ) ? '' : '<div class="cb-review-conclusion-title cb-block-header lr-cf">' . $args['content_title'] . '</div>';
		$content  .= empty( $args['content'] ) ? '' : '<div class="cb-review-conclusion lr-cf">' . $args['content'] . '</div>';

		$args['final_score'] = get_post_meta( $pid, '_lets_review_user_rating' . $uid, true ) ? get_post_meta( $pid, '_lets_review_user_rating' . $uid, true ) : 0;

		$args['animation'] = get_post_meta( $pid, '_lets_review_design_ani', true ) ? get_post_meta( $pid, '_lets_review_design_ani', true ) : 1;

		$args['format']       = empty( $args['format'] ) ? 1 : $args['format'];
		$args['design']       = empty( $args['design'] ) ? 1 : $args['design'];
		$args['design_skin']  = empty( $args['skin'] ) ? 1 : $args['skin'];
		$args['main_image']   = empty( $args['image'] ) ? '' : $args['image'];
		$args['custom_image'] = empty( $args['custom_image'] ) ? '' : $args['custom_image'];
		$args['icon_code']    = empty( $args['icon_code'] ) ? '' : html_entity_decode( $args['icon_code'] );

		if ( $args['design'] > 2 ) {
			$args['design'] = 2;
		}

		$args['accent'] = $this->lets_review_color( $pid );
		$args['type']   = 3;

		$img         = $this->lets_review_media_fi( $pid, '', $args );
		$user_rating = $this->lets_review_user_rating( $pid, $args );
		$f_score     = empty( $score_box ) ? '' : $this->lets_review_final_score( $pid, '', '', $args );

		$out_of_5 = '';
		$bg_style = '';

		if ( $args['format'] > 2 ) {
			$out_of_5 = ' cb-out-of-5';
			$bg_style = 'style="background-color: ' . $args['accent'] . ';"';
		}

		if ( empty( $img ) ) {
			$out_of_5 .= ' cb-no-fi';
		}
		switch ( $args['design'] ) {
			case 2:
				$output  = $img || $title ? '<div class="cb-review-block cb-review-pad cb-round-fi cb-title-block lr-cf" ' . $bg_style . '>' . $img . $title . '</div>' : '';
				$output .= $user_rating ? '<div class="cb-review-block">' . $user_rating . '</div>' : '';
				$output .= $content || $f_score ? '<div class="cb-review-block cb-review-pad cb-conclusion-block lr-cf">' . $f_score . $content . '</div>' : '';
				break;
			default:
				$output  = $img || $title ? '<div class="cb-review-block cb-review-pad cb-round-fi cb-summary-block lr-cf">' . $img . $title . '</div>' : '';
				$output .= $user_rating ? '<div class="cb-review-block cb-criteria-block cb-review-pad lr-cf">' . $user_rating . '</div>' : '';
				$output .= $f_score ? '<div class="cb-review-block cb-review-pad cb-conclusion-block cb-design-2 lr-cf">' . $f_score . $content . '</div>' : '';
				break;
		}

		return '<div id="lets-review-box-' . (int) $pid . '" class="cb-skin-base lets-review-box-wrap lets-review-ur-box cb-lr-font-p cb-design-' . $args['design'] . ' cb-skin-version-' . $args['design_skin'] . $out_of_5 . ' cb-score-type-' . $args['format'] . ' lets-review-ani-' . $args['animation'] . ' lr-cf" data-pid="' . (int) $pid . '" data-uid="' . (int) ( $uid ) . '" data-format="' . (int) ( $args['format'] ) . '" data-type="' . (int) ( $args['type'] ) . '">'
			. apply_filters( 'lets_review_user_rating_box_start', '' )
			. $output
			. apply_filters( 'lets_review_user_rating_box_end', '' )
			. '</div>';

	}

	function lets_review_new_review_box( $args ) {
		$pid = $args['pid'];
		if ( empty( $args['sc'] ) ) {
			$args['design']    = get_post_meta( $pid, '_lets_review_design', true );
			$args['crits']     = get_post_meta( $pid, '_lets_review_criterias', true );
			$args['affTitle']  = get_post_meta( $pid, '_lets_review_aff_title', true );
			$args['affBlocks'] = get_post_meta( $pid, '_lets_review_aff_buttons', true );
			$args['position']  = ! isset( $args['position'] ) ? 'center' : $args['position'];
			$source            = get_post_meta( $pid, '_lets_review_media_use_fi', true );
			$args['fiID']      = 1 == $source ? get_post_thumbnail_id( $pid ) : get_post_meta( $pid, '_lets_review_fi', true );
			$args['bgID']      = 1 == $source ? get_post_thumbnail_id( $pid ) : get_post_meta( $pid, '_lets_review_fi', true );
			$args['title']     = get_post_meta( $pid, '_lets_review_title_use_post', true ) == 1 ? get_the_title( $pid ) : get_post_meta( $pid, '_lets_review_title', true );
			$args['skin']      = get_post_meta( $pid, '_lets_review_design_skin', true );
			$args['aniType']   = get_post_meta( $pid, '_lets_review_design_ani', true );
			$args['scoreType'] = get_post_meta( $pid, '_lets_review_format', true );
			$args['iconCode']  = get_post_meta( $pid, '_lets_review_custom_icon', true );
			$args['imgIconID'] = get_post_meta( $pid, '_lets_review_custom_image', true );
			$accent            = get_post_meta( $pid, '_lets_review_color', true );
			if ( get_theme_mod( 'lr_override_accent_color_onoff' ) == true ) {
				$source = get_theme_mod( 'reviews_color_source', 1 );
				if ( 1 == $source ) {
					$categories_list = get_the_category( $pid );
					if ( ! empty( $categories_list ) ) {
						$accent = get_term_meta( $categories_list[0]->term_id, 'zeen_color', true ) != '' ? get_term_meta( $categories_list[0]->term_id, 'zeen_color', true ) : $accent;
					}
				} elseif ( 2 == $source ) {
					$accent = get_theme_mod( 'lr_override_accent_color', '#f8d92f' );
				}
			}
			$args['accent']          = $accent;
			$args['reviewType']      = get_post_meta( $pid, '_lets_review_type', true );
			$args['conclusionTitle'] = get_post_meta( $pid, '_lets_review_conclusion_title', true );
			$args['conclusion']      = get_post_meta( $pid, '_lets_review_conclusion', true );
			$args['prosTitle']       = get_post_meta( $pid, '_lets_review_pros_title', true );
			$args['consTitle']       = get_post_meta( $pid, '_lets_review_cons_title', true );
			$args['pros']            = get_post_meta( $pid, '_lets_review_pros', true );
			$args['cons']            = get_post_meta( $pid, '_lets_review_cons', true );
			$args['gallery']         = get_post_meta( $pid, '_lets_review_gallery_imgs', true );
			$args['galleryTitle']    = get_post_meta( $pid, '_lets_review_gallery_title', true );
			$args['scoreTitle']      = get_post_meta( $pid, '_lets_review_subtitle', true );
			$final_score             = get_post_meta( $pid, '_lets_review_final_score', true );
			$args['score']           = Lets_Review_Helpers::lets_review_get_convert_to_all_scores(
				array(
					'score'     => $final_score,
					'scoreType' => $args['scoreType'],
				)
			);
			if ( empty( $args['crits'] ) && empty( $final_score ) ) {
				$args['score'] = array();
			}
			$args['crits']        = empty( $args['crits'] ) ? array() : Lets_Review_Helpers::lets_review_get_convert_to_all_scores(
				array(
					'score'     => $args['crits'],
					'scoreType' => $args['scoreType'],
				)
			);
			$args['ratingsCount'] = get_post_meta( $pid, '_lets_review_user_rating_vote_count', true );
			$args['ratingScores'] = get_post_meta( $pid, '_lets_review_user_rating', true );

			if ( ! empty( $args['top_box'] ) && 5 != $args['location'] ) {
				$args['pros']            = '';
				$args['cons']            = '';
				$args['consTitle']       = '';
				$args['prosTitle']       = '';
				$args['title']           = '';
				$args['galleryTitle']    = '';
				$args['gallery']         = '';
				$args['conclusionTitle'] = '';
				$args['conclusion']      = '';
				$args['crits']           = '';
				$args['fiID']            = '';
			}
		} else {
			$args['ratingsCount'] = get_post_meta( $pid, '_lets_review_user_rating_vote_count' . $uid, true );
			$args['ratingScores'] = get_post_meta( $pid, '_lets_review_user_rating' . $uid, true );
			$args['custom_image'] = empty( $args['custom_image'] ) ? '' : $args['custom_image'];
			$args['icon_code']    = empty( $args['icon_code'] ) ? '' : html_entity_decode( $args['icon_code'] );
			$type                 = 2;
		}
		$args['gallery']        = empty( $args['gallery'] ) ? array() : $args['gallery'];
		$args['pros']           = empty( $args['pros'] ) ? array() : $args['pros'];
		$args['cons']           = empty( $args['cons'] ) ? array() : $args['cons'];
		$args['crits']          = empty( $args['crits'] ) || 3 == $args['reviewType'] ? array() : $args['crits'];
		$args['uid']            = empty( $args['uid'] ) ? $pid : $args['uid'];
		$args['showUserRating'] = 2 == $args['reviewType'] ? '' : true;
		$args['affBlocks']      = empty( $args['affBlocks'] ) ? array() : Lets_Review_Helpers::lets_review_cleanup_aff( $args['affBlocks'] );
		$args['pros']           = Lets_Review_Helpers::lets_review_cleanup_pros_cons( $args['pros'], 'positive' );
		$args['cons']           = Lets_Review_Helpers::lets_review_cleanup_pros_cons( $args['cons'], 'negative' );
		foreach ( $args['gallery'] as $key => $value ) {
			$args['gallery'][ $key ] = array(
				'id' => $value['attachment-id'],
			);
		}
		if ( empty( $args['accent'] ) ) {
			$args['accent'] = get_option( 'lets_review_gd_color', '#f8d92f' );
		}
		if ( 10 == $args['reviewType'] ) {
			return Lets_Review_Blocks::lets_review_comment_averages_block( $args );
		}
		return Lets_Review_Blocks::lets_review_block( $args );
	}

	/**
	 * Let's Review Frontend
	 *
	 * @since    1.0.0
	 */
	public function lets_review_append( $content, $pid = '' ) {
		global $post, $multipage, $numpages, $page;
		if ( ( ! ( is_singular() && in_the_loop() && is_main_query() ) && apply_filters( 'lets_review_output', true ) == '' ) || apply_filters( 'lets_review_append', true ) == '' || ( is_front_page() || is_archive() || is_date() || is_search() && get_option( 'lets_review_gen_load_outside_post', 1 ) == 0 ) || post_password_required() ) {
			return $content;
		}
		if ( ! empty( $_GET['tipi_builder_refresher'] ) ) {
			return $content;
		}
		if ( empty( $pid ) ) {
			$pid = $post->ID;
		}
		if ( empty( $pid ) ) {
			return $content;
		}
		if ( get_post_meta( $pid, '_lets_review_onoff', true ) != 1 ) {
			return $content;
		}

		$location   = get_post_meta( $pid, '_lets_review_location', true );
		$review_box = $this->lets_review_new_review_box( array( 'pid' => $pid ) );
		$top_review = $this->lets_review_new_review_box(
			array(
				'pid'      => $pid,
				'position' => 'left',
				'top_box'  => true,
				'location' => $location,
			)
		);
		if ( ! empty( $multipage ) ) {
			if ( $page == $numpages ) {
				if ( 1 == $location || 3 == $location ) {
					$content = $content . $review_box;
				}
			}
			if ( 1 == $page ) {
				if ( 2 == $location ) {
					$content = $review_box . $content;
				}
				if ( 3 == $location || 5 == $location ) {
					$content = $top_review . $content;
				}
			}
		} else {

			switch ( $location ) {
				case 2:
					$content = $review_box . $content;
					break;
				case 3:
					$content = $top_review . $content . $review_box;
					break;
				case 4:
					break;
				case 5:
					$content = $top_review . $content;
					break;
				default:
					$content = $content . $review_box;
					break;
			}
		}
		return $content;
	}

	public static function asset_optimizer() {
		$loader = apply_filters( 'lets_review_enqueue_assets_optimizer', true ) == true ? '' : true;
		if ( empty( $loader ) && is_singular() ) {
			$id = get_the_ID();
			if ( get_post_meta( $id, '_lets_review_onoff', true ) == 1 ) {
				$loader = true;
			} else {
				$block   = get_post( $id );
				$content = $block->post_content;
				if ( $block instanceof WP_Post ) {
					if ( false !== strpos( $content, '<!-- wp:lets-review' ) ) {
						$loader = true;
					}
				}
				if ( empty( $loader ) ) {
					if ( false !== strpos( $content, '[letsreview' ) ) {
						$loader = true;
					}
				}
			}
			if ( is_single() ) {
				$ipl = 2 === (int) get_post_meta( $id, 'zeen_next_post_auto_load', true ) ? '' : get_theme_mod( 'ipl' );
				if ( empty( $ipl ) && function_exists( 'ot_get_option' ) ) {
					$ipl = 2 === (int) get_post_meta( $id, 'codetipi_15zine_ipl', true ) ? '' : ot_get_option( 'ipl' );
					$ipl = 'on' == $ipl ? true : '';
				}
				if ( ! empty( $ipl ) ) {
					$loader = true;
				}
			}
		}
		return $loader;
	}

}
